#include "PulseOximeter.h"

//public variables
UINT8 Pox_HeartRate;
UINT8 Pox_SaturationValue;
UINT8 Pox_HeartBeatDetected;
UINT8 PoxGraph[POX_REAL_TIME_DATA_ARRAY_LENGTH];


/* Private functions */

//states
static void StateIdle(void);
static void StateMeasuring(void);

//functions
static void SaveGraphData(void);
static void ResetVariables(void);
static void CalculateHeartRateAndSpO2(void);

static void CalibrateIRLedIntensity(void);
static void CalibrateRedLedIntensity(void);

static void DoCalculations(void);
static void FindMaxAndMin(void);

//timer events
static void TimerSendGraphData_Event(void);


/* Main state machine */
const pFunc_t PoxStateMachine[] =
{
	StateIdle,
	StateMeasuring
};



typedef enum
{
	POX_CONTROL_RED,
	POX_CONTROL_IR,
};

/* Private variables */
static MovingAverage_uint16_t PoxRedLedSignal;
static MovingAverage_uint16_t PoxIRLedSignal;
static MovingAverage_uint16_t PoxRedBaselineSignal;
static MovingAverage_uint16_t PoxIRBaselineSignal;

static UINT16 PoxIRMax = 0;
static UINT16 PoxIRMin = 0;

static UINT16 PoxRedMax = 0;
static UINT16 PoxRedMin = 0;

static UINT8 PoxValidValue;

static UINT8 PoxHeartRateArray[POX_ARRAY_LENGTH];
static UINT8 PoxSpO2Array[POX_ARRAY_LENGTH];

UINT8 CopyOfPoxHeartRateArray[POX_ARRAY_LENGTH];
UINT8 CopyOfPoxSpO2Array[POX_ARRAY_LENGTH];



static UINT16 PoxSamplesBetweenPulses = 0; //counts the samples between 2 max points to calculate the HR


static UINT8 PoxActualState = POX_STATE_IDLE;
static UINT8 PoxActualSubState = 0;
static UINT8 PoxActualSubStateSubState = 0;

static UINT8 IsTimeToUpdateGraphData = FALSE;
static UINT8 PoxActualEvent = POX_EVENT_NONE;
static UINT8 PoxIsDiagnosticMode = FALSE;

static UINT8 IsSignalStable = FALSE;
static UINT16 TimerToRecalibrate = 0;


//timers
static UINT8 TimerSendGraphData;

				


//sending real time data

static UINT8 PoxGraphActualPosition;

static UINT16 PwmRedValue = 0;
static UINT16 PwmIRValue = 0;



#define POX_PWM_LEDS_CHANNEL	1		//PWM channel for LEDs
#define POX_PWM_OFFSET_VALUE	2		//PWM channel for signal offset



#define ADC_RESOLUTION			4095
#define REF_VOLTAGE_mV			3300	//mV

#define _1000mV		(UINT32)(((UINT32)1000*(UINT32)ADC_RESOLUTION)/(UINT32)REF_VOLTAGE_mV)
#define _500mV		(UINT32)(((UINT32)500*(UINT32)ADC_RESOLUTION)/(UINT32)REF_VOLTAGE_mV)
#define _100mV		(UINT32)(((UINT32)100*(UINT32)ADC_RESOLUTION)/(UINT32)REF_VOLTAGE_mV)
#define _50mV		(UINT32)(((UINT32)50*(UINT32)ADC_RESOLUTION)/(UINT32)REF_VOLTAGE_mV)



#define BASELINE_SETPOINT						_1000mV+_100mV+_100mV
#define BASELINE_SETPOINT_UPPER_LIMIT_MAX		(BASELINE_SETPOINT +_1000mV)
#define BASELINE_SETPOINT_UPPER_LIMIT_MID		(BASELINE_SETPOINT +_500mV)
#define BASELINE_SETPOINT_UPPER_LIMIT_MIN		(BASELINE_SETPOINT +_50mV)

#define BASELINE_SETPOINT_LOWER_LIMIT_MAX		(BASELINE_SETPOINT -_1000mV)
#define BASELINE_SETPOINT_LOWER_LIMIT_MID		(BASELINE_SETPOINT -_500mV)
#define BASELINE_SETPOINT_LOWER_LIMIT_MIN		(BASELINE_SETPOINT -_50mV)


#define PWM_MAX_VALUE	TPM_MAX_VAL


#define PWM_LARGE_STEP	50
#define PWM_MID_STEP	10
#define PWM_SHORT_STEP	1


#define IR_ON_TIME_us		50
#define IR_OFF_TIME_us		450

#define RED_ON_TIME_us		50
#define RED_OFF_TIME_us		450	



/*************************************************************/

#define Pwm_Start(dutyCycle)	Pwm1_Start(dutyCycle)
#define Pwm_Stop()				Pwm1_Stop()

#define START_TIMER_us()			TPM2_START()
#define STOP_TIMER_us()				TPM2_STOP()



/* States and substates */

static void StateIdle(void)
{
	//do nothing
}



static void StateMeasuring(void)
{
	UINT16 poxOutputSignalRaw, PoxBaselineSignalRaw;
	
	/* read POX signals: these are the steps:
		1. turn on IR LED
		2. wait 
		3. read baseline and read IR signal
		4. turn off IR LED (both LEDs off)
		5. wait 
		6. turn on Red LED
		7. wait 
		8. read Red signal
		9. turn off Red LED (both LEDs off)
		10. wait
	*/
	
	UINT16 elapsedTimeCopy = ElapsedTimeIn_us;

	switch (PoxActualSubStateSubState)
	{
		case 0:		
			POX_SET_SWITCH_CONTROL(POX_CONTROL_IR);			//set switch to select IR signals
			Pwm_Start(PwmIRValue);	//start pwm with IR value
			START_TIMER_us();								//start timer to wait 250us
			
//PTFD_PTFD2 = 1;			
			PoxActualSubStateSubState++;
			break;
			
		case 1:
			if (elapsedTimeCopy >= IR_ON_TIME_us)
			{			
				ElapsedTimeIn_us = 0;
//PTFD_PTFD2 = 0;
				//if (IsTimeToUpdateGraphData)
				{
					poxOutputSignalRaw =   ADC_Read12b(POX_ADC_CHANNEL_OUTPUT_SIGNAL);	//read IR baseline and IR signal
					PoxBaselineSignalRaw = ADC_Read12b(POX_ADC_CHANNEL_BASELINE_SIGNAL);
									
					MovingAverage_PushNewValue16b(&PoxIRBaselineSignal, PoxBaselineSignalRaw);	//store IR baseline and IR signal
					MovingAverage_PushNewValue16b(&PoxIRLedSignal, poxOutputSignalRaw);
				}				
				
				
				Pwm_Stop();		//turn off LEDs									
				PoxActualSubStateSubState++;
			}
			break;
	
		case 2:
			if (elapsedTimeCopy >= IR_OFF_TIME_us)
			{
				ElapsedTimeIn_us = 0;
//PTFD_PTFD2 = 1;

				POX_SET_SWITCH_CONTROL(POX_CONTROL_RED);	//set switch to select Red signal								
				Pwm_Start(PwmRedValue);	//start pwm for Red LED
				PoxActualSubStateSubState++;					
			}
			break;
			
		case 3:
			if (elapsedTimeCopy >= RED_ON_TIME_us)
			{				
				ElapsedTimeIn_us = 0;
//PTFD_PTFD2 = 0;				
				//if (IsTimeToUpdateGraphData)
				{					
					poxOutputSignalRaw = ADC_Read12b(POX_ADC_CHANNEL_OUTPUT_SIGNAL);		//Read RED signal
					PoxBaselineSignalRaw = ADC_Read12b(POX_ADC_CHANNEL_BASELINE_SIGNAL);	////Read RED baseline				

					MovingAverage_PushNewValue16b(&PoxRedBaselineSignal, PoxBaselineSignalRaw);	//store Red baseline and red signal
					MovingAverage_PushNewValue16b(&PoxRedLedSignal, poxOutputSignalRaw);
				}

				Pwm_Stop();		//turn off LEDs
		
				PoxActualSubStateSubState++;
			}
			break;
			
		case 4:
			if (elapsedTimeCopy >= RED_OFF_TIME_us)
			{			
				ElapsedTimeIn_us = 0;
				PoxActualSubStateSubState = 0;				
				PoxSamplesBetweenPulses++;
							
				if (IsTimeToUpdateGraphData && PoxIsDiagnosticMode)
				{					
					IsTimeToUpdateGraphData = FALSE;										
					SaveGraphData();
					SwTimer_StartTimer(TimerSendGraphData, POX_UPDATE_GRAPH_PERIOD);
				}

				#define RECALIBRATE_TIME	512
									
				if (TimerToRecalibrate == 0)
				{
					static toogle = 0;
					
					#define MAX_CALIBRATION_ATTEMPTS	128
					static UINT16 calibrationAttempts = 0;
														
					//check if both baselines are within the limits					
					if 	((PoxIRBaselineSignal.Result > BASELINE_SETPOINT_UPPER_LIMIT_MIN)  || (PoxIRBaselineSignal.Result < BASELINE_SETPOINT_LOWER_LIMIT_MIN) ||
						((PoxRedBaselineSignal.Result > BASELINE_SETPOINT_UPPER_LIMIT_MIN) || (PoxRedBaselineSignal.Result < BASELINE_SETPOINT_LOWER_LIMIT_MIN)))
					{
						//IsSignalStable = FALSE;

						if (toogle == POX_CONTROL_RED)
						{
							if ((PoxRedBaselineSignal.Result > BASELINE_SETPOINT_UPPER_LIMIT_MID) || (PoxRedBaselineSignal.Result < BASELINE_SETPOINT_LOWER_LIMIT_MID))
							{
								CalibrateRedLedIntensity();						
								TimerToRecalibrate = 16;
								
								//if (calibrationAttempts < 0xFF)
								{
									calibrationAttempts++;
								}
							}
							toogle = POX_CONTROL_IR;							
						}
						else if (toogle == POX_CONTROL_IR)
						{							
							if ((PoxIRBaselineSignal.Result > BASELINE_SETPOINT_UPPER_LIMIT_MID)  || (PoxIRBaselineSignal.Result < BASELINE_SETPOINT_LOWER_LIMIT_MID))
							{
								CalibrateIRLedIntensity();
								TimerToRecalibrate = 16;

								//if (calibrationAttempts < 0xFF)
								{
									calibrationAttempts++;
								}							
							}
							toogle = POX_CONTROL_RED;
						}
					}
					else
					{					
						TimerToRecalibrate = RECALIBRATE_TIME;
						calibrationAttempts = 0;
						IsSignalStable = TRUE;
					}
										
					if (calibrationAttempts > MAX_CALIBRATION_ATTEMPTS)
					{
						IsSignalStable = FALSE;
					}
					else
					{
						IsSignalStable = TRUE;
					}
				}
				else
				{
					TimerToRecalibrate--;					
				}
				
				
				//if (IsSignalStable)
				{
					FindMaxAndMin();
				}
				
				if ((PoxValidValue >= 10) && (!PoxIsDiagnosticMode))
				{
					//generate pox ok event
					PoxActualEvent = POX_EVENT_MEASUREMENT_COMPLETE_OK;
					Pox_AbortMeasurement();
				}
			}
			break;
	}
}









static void CalibrateRedLedIntensity(void)
{
	//if signal is larger than setpoint
	if (PoxRedBaselineSignal.Result > BASELINE_SETPOINT_UPPER_LIMIT_MAX)
	{
		if (PwmRedValue > PWM_LARGE_STEP)
		{
			//decrease offset by large steps
			PwmRedValue -= PWM_LARGE_STEP;			
		}
		else if (PwmRedValue > PWM_MID_STEP)
		{
			//decrease offset by large steps
			PwmRedValue -= PWM_MID_STEP;			
		}
		else if (PwmRedValue > PWM_SHORT_STEP)
		{
			//decrease offset by large steps
			PwmRedValue -= PWM_SHORT_STEP;			
		}
		//else PwmRedValue = PWM_MAX_VALUE;
	}
	else if (PoxRedBaselineSignal.Result > BASELINE_SETPOINT_UPPER_LIMIT_MID)
	{
		//decrease offset by midium steps
		if (PwmRedValue > PWM_MID_STEP)
		{
			//decrease offset by large steps
			PwmRedValue -= PWM_MID_STEP;			
		}
		else if (PwmRedValue > PWM_SHORT_STEP)
		{
			//decrease offset by large steps
			PwmRedValue -= PWM_SHORT_STEP;			
		}		
	}
	else if (PoxRedBaselineSignal.Result > BASELINE_SETPOINT_UPPER_LIMIT_MIN)
	{
		//decrease offset by short steps		
		if (PwmRedValue > PWM_SHORT_STEP)
		{
			//decrease offset by large steps
			PwmRedValue -= PWM_SHORT_STEP;			
		}
	}
	
	
	//if signal is lower than setpoint	
	
	else if (PoxRedBaselineSignal.Result < BASELINE_SETPOINT_LOWER_LIMIT_MAX)
	{
		//increase offset by large steps
		if (PwmRedValue < (PWM_MAX_VALUE - PWM_LARGE_STEP))
		{
			//decrease offset by large steps
			PwmRedValue += PWM_LARGE_STEP;
		}	
		else if (PwmRedValue < (PWM_MAX_VALUE - PWM_MID_STEP))
		{
			//decrease offset by large steps
			PwmRedValue += PWM_MID_STEP;
		}
		else if (PwmRedValue < (PWM_MAX_VALUE - PWM_SHORT_STEP))
		{
			//decrease offset by large steps
			PwmRedValue += PWM_SHORT_STEP;			
		}
		//else PwmRedValue = 0;
		
	}
	else if (PoxRedBaselineSignal.Result < BASELINE_SETPOINT_LOWER_LIMIT_MIN)
	{
		//increase offset by medium steps
		if (PwmRedValue < (PWM_MAX_VALUE - PWM_MID_STEP))
		{
			//decrease offset by large steps
			PwmRedValue += PWM_MID_STEP;
		}	
		else if (PwmRedValue < (PWM_MAX_VALUE - PWM_SHORT_STEP))
		{
			//decrease offset by large steps
			PwmRedValue += PWM_SHORT_STEP;
		}	
	}
	else if (PoxRedBaselineSignal.Result < BASELINE_SETPOINT_LOWER_LIMIT_MID)
	{
		//increase offset by short steps		
		if (PwmRedValue < (PWM_MAX_VALUE - PWM_SHORT_STEP))
		{
			//decrease offset by large steps
			PwmRedValue += PWM_SHORT_STEP;
		}
	}	
}


	
static void CalibrateIRLedIntensity(void)
{
	//if signal is larger than setpoint
	if (PoxIRBaselineSignal.Result > BASELINE_SETPOINT_UPPER_LIMIT_MAX)
	{
		if (PwmIRValue > PWM_LARGE_STEP)
		{
			//decrease offset by large steps
			PwmIRValue -= PWM_LARGE_STEP;			
		}
		else if (PwmIRValue > PWM_MID_STEP)
		{
			//decrease offset by large steps
			PwmIRValue -= PWM_MID_STEP;			
		}
		else if (PwmIRValue > PWM_SHORT_STEP)
		{
			//decrease offset by large steps
			PwmIRValue -= PWM_SHORT_STEP;			
		}
		//else PwmIRValue = PWM_MAX_VALUE;
	}
	else if (PoxIRBaselineSignal.Result > BASELINE_SETPOINT_UPPER_LIMIT_MID)
	{
		//decrease offset by midium steps
		if (PwmIRValue > PWM_MID_STEP)
		{
			//decrease offset by large steps
			PwmIRValue -= PWM_MID_STEP;			
		}
		else if (PwmIRValue > PWM_SHORT_STEP)
		{
			//decrease offset by large steps
			PwmIRValue -= PWM_SHORT_STEP;			
		}		
	}
	else if (PoxIRBaselineSignal.Result > BASELINE_SETPOINT_UPPER_LIMIT_MIN)
	{
		//decrease offset by short steps		
		if (PwmIRValue > PWM_SHORT_STEP)
		{
			//decrease offset by large steps
			PwmIRValue -= PWM_SHORT_STEP;			
		}
	}
	
	
	//if signal is lower than setpoint	
	
	else if (PoxIRBaselineSignal.Result < BASELINE_SETPOINT_LOWER_LIMIT_MAX)
	{
		//increase offset by large steps
		if (PwmIRValue < (PWM_MAX_VALUE - PWM_LARGE_STEP))
		{
			//decrease offset by large steps
			PwmIRValue += PWM_LARGE_STEP;
		}
		else if (PwmIRValue < (PWM_MAX_VALUE - PWM_MID_STEP))
		{
			//decrease offset by large steps
			PwmIRValue += PWM_MID_STEP;
		}
		else if (PwmIRValue < (PWM_MAX_VALUE - PWM_SHORT_STEP))
		{
			//decrease offset by large steps
			PwmIRValue += PWM_SHORT_STEP;
		}
		//else PwmIRValue = 0;
	}
	else if (PoxIRBaselineSignal.Result < BASELINE_SETPOINT_LOWER_LIMIT_MIN)
	{
		//increase offset by medium steps
		if (PwmIRValue < (PWM_MAX_VALUE - PWM_MID_STEP))
		{
			//decrease offset by large steps
			PwmIRValue += PWM_MID_STEP;
		}
		else if (PwmIRValue < (PWM_MAX_VALUE - PWM_SHORT_STEP))
		{
			//decrease offset by large steps
			PwmIRValue += PWM_SHORT_STEP;
		}	
	}
	else if (PoxIRBaselineSignal.Result < BASELINE_SETPOINT_LOWER_LIMIT_MID)
	{
		//increase offset by short steps		
		if (PwmIRValue < (PWM_MAX_VALUE - PWM_SHORT_STEP))
		{
			//decrease offset by large steps
			PwmIRValue += PWM_SHORT_STEP;
		}
	}		
}


typedef enum
{
	
	POX_SUBSTATE_FINDING_MAX,
	POX_SUBSTATE_FINDING_MIN
};


//the max freq we want to detect is 4Hz (250ms) (a person with tachicardia)
//we are sampling signal every 1ms so the comparation window must be 250 samples (250ms)

#define COMPARATION_WINDOW_INTERVAL		250			//samples between max and min


//TO DO: Add watchdog timer

UINT8 maxCounter = 0;
UINT8 minCounter = 0;
UINT16 IRPeakToPeakVoltage = 0;
UINT16 RedPeakToPeakVoltage = 0;
UINT16 IrRms = 0;
UINT16 RedRms = 0;

const UINT16 Log10LookUpTable[] = 
{
	0, 301, 602, 778, 903, 1000, 1079, 1146, 1204, 1255, 1301, 1342, 1380, 1414, 1447, 1477, 1505, 1531, 1556, 1579, 1602, 1623, 1643, 1662, 1681, 1698, 1716, 1732, 1748, 1763, 1778, 1792, 1806, 1819, 1832, 1845, 1857, 1869, 1880, 1892, 1903, 1913, 1924, 1934, 1944, 1954, 1963, 1973, 1982, 1991, 2000, 2008, 2017, 2025, 2033, 2041, 2049, 2056, 2064, 2071, 2079, 2086, 2093, 2100, 2107, 2113, 2120, 2127, 2133, 2139, 2146, 2152, 2158, 2164, 2170, 2176, 2181, 2187, 2193, 2198, 2204, 2209, 2214, 2220, 2225, 2230, 2235, 2240, 2245, 2250, 2255, 2260, 2264, 2269, 2274, 2278, 2283, 2287, 2292, 2296, 2301, 2305, 2309, 2313, 2318, 2322, 2326, 2330, 2334, 2338, 2342, 2346, 2350, 2354, 2357, 2361, 2365, 2369, 2372, 2376, 2380, 2383, 2387, 2390, 2394, 2397, 2401, 2404, 2408, 2411, 2414, 2418, 2421, 2424, 2428, 2431, 2434, 2437, 2440, 2444, 2447, 2450, 2453, 2456, 2459, 2462, 2465, 2468, 2471, 2474, 2477, 2480, 2482, 2485, 2488, 2491, 2494, 2496, 2499, 2502, 2505, 2507, 2510, 2513, 2515, 2518, 2521, 2523, 2526, 2528, 2531, 2534, 2536, 2539, 2541, 2544, 2546, 2549, 2551, 2553, 2556, 2558, 2561, 2563, 2565, 2568, 2570, 2572, 2575, 2577, 2579, 2582, 2584, 2586, 2588, 2591, 2593, 2595, 2597, 2599, 2602, 2604, 2606, 2608, 2610, 2612, 2614, 2617, 2619, 2621, 2623, 2625, 2627, 2629, 2631, 2633, 2635, 2637, 2639, 2641, 2643, 2645, 2647, 2649, 2651, 2653, 2655, 2657, 2658, 2660, 2662, 2664, 2666, 2668, 2670, 2672, 2673, 2675, 2677, 2679, 2681, 2683, 2684, 2686, 2688, 2690, 2691, 2693, 2695, 2697, 2698, 2700, 2702, 2704, 2705, 2707, 2709, 2710, 2712, 2714, 2716, 2717, 2719, 2720, 2722, 2724, 2725, 2727, 2729, 2730, 2732, 2733, 2735, 2737, 2738, 2740, 2741, 2743, 2745, 2746, 2748, 2749, 2751, 2752, 2754, 2755, 2757, 2758, 2760, 2761, 2763, 2764, 2766, 2767, 2769, 2770, 2772, 2773, 2775, 2776, 2778, 2779, 2781, 2782, 2783, 2785, 2786, 2788, 2789, 2790, 2792, 2793, 2795, 2796, 2797, 2799, 2800, 2802, 2803, 2804, 2806, 2807, 2808, 2810, 2811, 2812, 2814, 2815, 2816, 2818, 2819, 2820, 2822, 2823, 2824, 2826, 2827, 2828, 2829, 2831, 2832, 2833, 2835, 2836, 2837, 2838, 2840, 2841, 2842, 2843, 2845, 2846, 2847, 2848, 2850, 2851, 2852, 2853, 2854, 2856, 2857, 2858, 2859, 2860, 2862, 2863, 2864, 2865, 2866, 2868, 2869, 2870, 2871, 2872, 2873, 2875, 2876, 2877, 2878, 2879, 2880, 2881, 2883, 2884, 2885, 2886, 2887, 2888, 2889, 2890, 2892, 2893, 2894, 2895, 2896, 2897, 2898, 2899, 2900, 2902, 2903, 2904, 2905, 2906, 2907, 2908, 2909, 2910, 2911, 2912, 2913, 2914, 2915, 2916, 2918, 2919, 2920, 2921, 2922, 2923, 2924, 2925, 2926, 2927, 2928, 2929, 2930, 2931, 2932, 2933, 2934, 2935, 2936, 2937, 2938, 2939, 2940, 2941, 2942, 2943, 2944, 2945, 2946, 2947, 2948, 2949, 2950, 2951, 2952, 2953, 2954, 2955, 2956, 2957, 2958, 2959, 2959, 2960, 2961, 2962, 2963, 2964, 2965, 2966, 2967, 2968, 2969, 2970, 2971, 2972, 2973, 2974, 2974, 2975, 2976, 2977, 2978, 2979, 2980, 2981, 2982, 2983, 2984, 2984, 2985, 2986, 2987, 2988, 2989, 2990, 2991, 2992, 2992, 2993, 2994, 2995, 2996, 2997, 2998, 2999, 3000, 3000, 3001, 3002, 3003, 3004, 3005, 3006, 3006, 3007, 3008, 3009
};


static void FindMaxAndMin(void)
{
	static UINT16 comparationWindowTimer;
	static UINT8 subStateFindingMaxAndMin;
		
	
	if (subStateFindingMaxAndMin == POX_SUBSTATE_FINDING_MAX)
	{
		//if the Pox signal is going up
		if (PoxIRLedSignal.Result > PoxIRMax)
		{
			PoxIRMax = PoxIRLedSignal.Result;							//store new max values
			PoxRedMax = PoxRedLedSignal.Result;
			
			comparationWindowTimer = COMPARATION_WINDOW_INTERVAL;		//restart counter				
		}
		else
		{	
			//slope changed, now signal is going down
			comparationWindowTimer--;
			
			if (comparationWindowTimer == 0)
			{
				//no more max peaks detected in the window time, go to detection of min
				subStateFindingMaxAndMin = POX_SUBSTATE_FINDING_MIN;
				maxCounter++;
				PoxIRMin = PoxIRMax;
				PoxRedMin = PoxRedMax;
			}
		}
	}
	else if (subStateFindingMaxAndMin == POX_SUBSTATE_FINDING_MIN)
	{
		//if the Pox signal is going down
		if (PoxIRLedSignal.Result < PoxIRMin)
		{
			PoxIRMin = PoxIRLedSignal.Result;							//store new min values
			PoxRedMin = PoxRedLedSignal.Result;
			comparationWindowTimer = COMPARATION_WINDOW_INTERVAL;		//restart counter
		}
		else
		{	
			//slope changed, now signal is going up
			comparationWindowTimer--;
			
			if (comparationWindowTimer == 0)
			{
				//no more min peaks detected in the window time, go to next state
				subStateFindingMaxAndMin = POX_SUBSTATE_FINDING_MAX;					
				minCounter++;

				DoCalculations();		//do calculations before updating min and max values
				
				PoxIRMax = PoxIRMin;
				PoxRedMax = PoxRedMin;
				Pox_HeartBeatDetected = TRUE;				
			}
		}
	}
}


static void DoCalculations(void)
{
	UINT8 poxInstantaneousHeartRate;
	UINT8 poxInstantaneousSpO2;
	UINT8 i=0;
	
	UINT32 numerator;
	UINT16 lookUpTableIndex = 0;
	
	
	IRPeakToPeakVoltage = PoxIRMax - PoxIRMin;		//calculate peak to peak voltage
	RedPeakToPeakVoltage = PoxRedMax - PoxRedMin;
	
	if (IRPeakToPeakVoltage < 2000 && RedPeakToPeakVoltage < 2000)
	{
		//calculate Vrms = 0.5*Vpp/sqrt(2)
		numerator = (UINT32)((UINT32)IRPeakToPeakVoltage * 707);
		IrRms = (UINT16)((UINT32)(numerator)/(UINT32)(2*1000));
		
		numerator = (UINT32)((UINT32)RedPeakToPeakVoltage * 707);
		RedRms = (UINT16)((UINT32)(numerator)/(UINT32)(2*1000));
		
		if (RedRms < 1024 && IrRms < 1024)
		{					
			lookUpTableIndex = (UINT16)(RedRms/2);
			numerator = (UINT32)((UINT32)Log10LookUpTable[lookUpTableIndex]*(UINT32)100);
			
			lookUpTableIndex = (UINT16)(IrRms/2);

			poxInstantaneousSpO2 = (UINT8) ((UINT32)(numerator)/(UINT32)Log10LookUpTable[lookUpTableIndex]);
			
			if (poxInstantaneousSpO2 > 100)
			{
				poxInstantaneousSpO2 = 100;
			}
			
			//Calculate instantaneous HR
			poxInstantaneousHeartRate = (UINT8) (60000 / (PoxSamplesBetweenPulses));// * POX_SAMPLING_PERIOD));
			
			//Shift samples of FIFO	
			for (i=OLDEST_ELEMENT; i>NEWEST_ELEMENT; i--)
			{
				PoxHeartRateArray[i] = PoxHeartRateArray[i-1];	
				PoxSpO2Array[i] = PoxSpO2Array[i-1];
			}
			
			//insert new sample into FIFO
			PoxHeartRateArray[NEWEST_ELEMENT] = poxInstantaneousHeartRate;
			PoxSpO2Array[NEWEST_ELEMENT] = poxInstantaneousSpO2;				
					
			CalculateHeartRateAndSpO2();		//calculate average of medians
			
			PoxSamplesBetweenPulses = 0;		//Restart samples counter 									
		}
	/*	else
		{
			Pox_HeartRate = 0;
			Pox_SaturationValue = 0;
		}
	*/
	}
	/*else
	{
		Pox_HeartRate = 0;
		Pox_SaturationValue = 0;
	}
	*/
}


static void CalculateHeartRateAndSpO2(void)
//order HearRate values in array and average samples in the middle		
{
	UINT8 startIndex;
	UINT8 smallestIndex;
	UINT8 currentIndex;
	UINT8 tempStoreValue;
	UINT16 sum;
	UINT8 i;
	
	//Create a copy of the arrays
	for (i=0; i<POX_ARRAY_LENGTH; i++)
	{
		CopyOfPoxHeartRateArray[i] = PoxHeartRateArray[i];		
		CopyOfPoxSpO2Array[i] = PoxSpO2Array[i];
	}
	
	// Order array values in ascending order 
	for (startIndex = 0; startIndex < POX_ARRAY_LENGTH; startIndex++)
	{
		smallestIndex = startIndex;

		for (currentIndex = startIndex + 1; currentIndex < POX_ARRAY_LENGTH; currentIndex++)
		{
			if (CopyOfPoxHeartRateArray[currentIndex] < CopyOfPoxHeartRateArray[smallestIndex])
			{
				smallestIndex = currentIndex;
			}
		}

		tempStoreValue = (UINT8) CopyOfPoxHeartRateArray[startIndex];

		CopyOfPoxHeartRateArray[startIndex] = CopyOfPoxHeartRateArray[smallestIndex];
		CopyOfPoxHeartRateArray[smallestIndex] = tempStoreValue;
	}						
	
	// Calculate HR averaging values from 2-5 (avoids using false HR values)
	//
	//	 		 	                  		  [0,1,2,3,4,5,6,7] 
	//										   | |         | |
	//										   | |         | |
	//									remove - -         - -
	
	sum = 0;
	
	for (i = 2; i < 6; i++)
	{
		sum += CopyOfPoxHeartRateArray[i];
	}

	Pox_HeartRate = (UINT8)(sum / 4);
	
	
	//Same procedure, but with SpO2 array
	
	
	// Order array values in ascending order 
	for (startIndex = 0; startIndex < POX_ARRAY_LENGTH; startIndex++)
	{
		smallestIndex = startIndex;

		for (currentIndex = startIndex + 1; currentIndex < POX_ARRAY_LENGTH; currentIndex++)
		{
			if (CopyOfPoxSpO2Array[currentIndex] < CopyOfPoxSpO2Array[smallestIndex])
			{
				smallestIndex = currentIndex;
			}
		}

		tempStoreValue = (UINT8) CopyOfPoxSpO2Array[startIndex];

		CopyOfPoxSpO2Array[startIndex] = CopyOfPoxSpO2Array[smallestIndex];
		CopyOfPoxSpO2Array[smallestIndex] = tempStoreValue;
	}						
	
	// Calculate SpO2 averaging values from 2-5 (avoids using false values)
	//
	//	 		 	                  		  [0,1,2,3,4,5,6,7] 
	//										   | |         | |
	//										   | |         | |
	//									remove - -         - -
	
	sum = 0;
	
	for (i = 2; i < 6; i++)
	{
		sum += CopyOfPoxSpO2Array[i];
	}

	Pox_SaturationValue = (UINT8)(sum / 4);
	
	PoxValidValue++;
	
}


static void ResetVariables(void)
{
	UINT8 i = 0;
	
	//Reset variables
	PoxIRMax = 0;
	PoxIRMin = 4096;
	
	PoxRedMax = 0;
	PoxRedMin = 4096;

	PoxSamplesBetweenPulses = 0;
	
	PwmRedValue = 0;
	PwmIRValue = 0;
	
	PoxValidValue = 0;
	
	PoxActualSubState = 0;	
	PoxActualSubStateSubState = 0;	
		
	IsTimeToUpdateGraphData = FALSE;	
	ElapsedTimeIn_us = 0;		
	TimerToRecalibrate = 0;
	
	IsSignalStable = FALSE;

	//clean arrays
	for (i=0; i<POX_ARRAY_LENGTH; i++)
	{
		CopyOfPoxHeartRateArray[i] = 0;
		CopyOfPoxSpO2Array[i] = 0;
		
		PoxHeartRateArray[i] = 0;
		PoxSpO2Array[i] = 0;
	}
}


#define ZERO_VALUE		2024

#ifdef POX_DEBUG
static void SaveGraphData(void)
{
	//store data and send it when the buffer is full
	if (PoxGraphActualPosition < POX_REAL_TIME_DATA_ARRAY_LENGTH)
	{
		//if(IsSignalStable)
		{			
			//send PoxSignal
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(PoxIRLedSignal.Result >> 8); 		//higher byte
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(PoxIRLedSignal.Result & 0x00FF); 	//lower byte
			
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(PoxRedLedSignal.Result >> 8); 		//higher byte
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(PoxRedLedSignal.Result & 0x00FF); 	//lower byte
		}
		/*else
		{
			//send PoxSignal
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(ZERO_VALUE >> 8); 		//higher byte
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(ZERO_VALUE & 0x00FF); 	//lower byte
			
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(ZERO_VALUE >> 8); 		//higher byte
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(ZERO_VALUE & 0x00FF); 	//lower byte
			
		}*/
	}
	else
	{
		//buffer is full, call the data ready event
		//PoxActualEvent = POX_EVENT_NEW_DATA_READY;
		PoxActualEvent = POX_EVENT_DEBUG_MODE_NEW_DATA_READY;
		PoxGraphActualPosition = 0;
	}
}

#else

#define AMPLIFY_SIGNAL

static void SaveGraphData(void)
{	
	//store data and send it when the buffer is full
	if (PoxGraphActualPosition < POX_REAL_TIME_DATA_ARRAY_LENGTH)
	{
		//if(IsSignalStable)
		{			
			//send PoxSignal
			#ifdef AMPLIFY_SIGNAL

			INT16 dataToSend = 0;
			UINT16 dataSigned = 0;
			
			dataToSend = (INT16)(PoxIRLedSignal.Result - 2047);
			dataToSend = dataToSend*4;
						
			dataToSend += 2047;
			
			if (dataToSend > 4095)
			{
				dataToSend = 4095;
			}
			else if (dataToSend < 0)
			{
				dataToSend = 0;
			}

			dataSigned = (UINT16)(dataToSend);
			
			
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(dataSigned >> 8); 		//higher byte
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(dataSigned & 0x00FF); 	//lower byte
			
			#else			
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(PoxIRLedSignal.Result >> 8); 		//higher byte
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(PoxIRLedSignal.Result & 0x00FF); 	//lower byte
			#endif
		}/*
		else
		{
			//send PoxSignal
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(ZERO_VALUE >> 8); 		//higher byte
			PoxGraph[PoxGraphActualPosition++] = (UINT8)(ZERO_VALUE & 0x00FF); 	//lower byte						
		}*/
	}
	else
	{
		//buffer is full, call the data ready event
		PoxActualEvent = POX_EVENT_NEW_DATA_READY;		
		PoxGraphActualPosition = 0;
	}
}
#endif

static void TimerSendGraphData_Event(void)
{
	IsTimeToUpdateGraphData = TRUE;
}



void Pox_Init(void)
{	
	TimerSendGraphData = SwTimer_CreateTimer(TimerSendGraphData_Event);	//create timer
}


void Pox_PeriodicTask(void)
{
	/* State machine handler */
	PoxStateMachine[PoxActualState]();

	/* Event handler */
	if (PoxActualEvent != POX_EVENT_NONE)
	{
		if (Pox_Events[PoxActualEvent] != NULL)
		{
			Pox_Events[PoxActualEvent]();	//execute registered event
			PoxActualEvent = POX_EVENT_NONE;
		}
	}
	
}


UINT8 Pox_StartMeasurement(void)
{
	UINT8 status = FALSE;

	if (PoxActualState == POX_STATE_IDLE)
	{
		ADC_Init12b(1 << POX_ADC_CHANNEL_OUTPUT_SIGNAL |
		            1 << POX_ADC_CHANNEL_BASELINE_SIGNAL);
		
		POX_INIT_SWITCH_CONTROL_PIN();		//Init GPIOs
		
		Pwm1_Init();
		Tpm2_Init();
	
		PoxIsDiagnosticMode = FALSE;
		PoxActualState = POX_STATE_MEASURING;
		ResetVariables();		
	
		SwTimer_StartTimer(TimerSendGraphData, POX_UPDATE_GRAPH_PERIOD);	//start timer to sample ADC
		status = TRUE;			
	}
	else
	{
		status = FALSE;
	}

	return status;	
}


void Pox_AbortMeasurement(void)
{
	PoxActualState = POX_STATE_IDLE;
	SwTimer_StopTimer(TimerSendGraphData);
	
	STOP_TIMER_us();				
	Pwm_Stop();		//turn off LEDs
}


UINT8 Pox_DiagnosticModeStartMeasurement(void)
{
	UINT8 status = FALSE;

	if (PoxActualState == POX_STATE_IDLE)
	{
		ADC_Init12b(1 << POX_ADC_CHANNEL_OUTPUT_SIGNAL |
		            1 << POX_ADC_CHANNEL_BASELINE_SIGNAL);

		POX_INIT_SWITCH_CONTROL_PIN();		//Init GPIOs
		
		Pwm1_Init();
		Tpm2_Init();

		PoxIsDiagnosticMode = TRUE;
		PoxActualState = POX_STATE_MEASURING;
		ResetVariables();		
	
		SwTimer_StartTimer(TimerSendGraphData, POX_UPDATE_GRAPH_PERIOD);	//start timer to sample ADC
		status = TRUE;
		
	}
	else
	{
		status = FALSE;
	}

	return status;
}


void Pox_DiagnosticModeStopMeasurement(void)
{
	Pox_AbortMeasurement();
}





